'use client'

import React, { useState, useEffect } from 'react'
import { Client, CallLog, CreateCallLogRequest } from '@/types'
import { 
  XMarkIcon,
  PhoneIcon,
  ClockIcon,
  CheckCircleIcon,
  XCircleIcon,
  ExclamationTriangleIcon,
  PlayIcon,
  StopIcon
} from '@heroicons/react/24/outline'

interface CallLogModalProps {
  isOpen: boolean
  onClose: () => void
  client: Client
  existingCallLog?: CallLog
  onSave: (callLog: CreateCallLogRequest) => Promise<void>
}

export default function CallLogModal({ 
  isOpen, 
  onClose, 
  client, 
  existingCallLog, 
  onSave 
}: CallLogModalProps) {
  const [formData, setFormData] = useState<CreateCallLogRequest>({
    client_id: client.id,
    call_type: 'outbound',
    call_status: 'completed',
    call_duration: 0,
    notes: '',
    callback_requested: false,
    callback_time: '',
  })

  const [isCallActive, setIsCallActive] = useState(false)
  const [callStartTime, setCallStartTime] = useState<Date | null>(null)
  const [callDuration, setCallDuration] = useState(0)
  const [errors, setErrors] = useState<Record<string, string>>({})
  const [isLoading, setIsLoading] = useState(false)

  // Timer for active call
  useEffect(() => {
    let interval: NodeJS.Timeout | null = null
    
    if (isCallActive && callStartTime) {
      interval = setInterval(() => {
        const now = new Date()
        const duration = Math.floor((now.getTime() - callStartTime.getTime()) / 1000)
        setCallDuration(duration)
        setFormData(prev => ({ ...prev, call_duration: duration }))
      }, 1000)
    }

    return () => {
      if (interval) clearInterval(interval)
    }
  }, [isCallActive, callStartTime])

  // Initialize form with existing data
  useEffect(() => {
    if (existingCallLog) {
      setFormData({
        client_id: existingCallLog.client_id,
        call_type: existingCallLog.call_type,
        call_status: existingCallLog.call_status,
        call_duration: existingCallLog.call_duration || 0,
        notes: existingCallLog.notes,
        callback_requested: existingCallLog.callback_requested,
        callback_time: existingCallLog.callback_time || '',
      })
      setCallDuration(existingCallLog.call_duration || 0)
    } else {
      // Reset form for new call log
      setFormData({
        client_id: client.id,
        call_type: 'outbound',
        call_status: 'completed',
        call_duration: 0,
        notes: '',
        callback_requested: false,
        callback_time: '',
      })
      setCallDuration(0)
    }
    setIsCallActive(false)
    setCallStartTime(null)
    setErrors({})
  }, [existingCallLog, client.id, isOpen])

  const formatDuration = (seconds: number) => {
    const mins = Math.floor(seconds / 60)
    const secs = seconds % 60
    return `${mins}:${secs.toString().padStart(2, '0')}`
  }

  const startCall = async () => {
    try {
      const token = localStorage.getItem('token')
      if (!token) {
        alert('Authentication required. Please log in again.')
        return
      }

      // Check for conflicts and start call status
      const response = await fetch('/api/user-status', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({
          action: 'start_call',
          client_id: client.id
        })
      })

      if (!response.ok) {
        const errorData = await response.json()
        if (response.status === 409) {
          // Conflict - show user-friendly error
          alert(errorData.error)
          return
        } else {
          throw new Error(errorData.error || 'Failed to start call')
        }
      }

      // Call started successfully
      setIsCallActive(true)
      setCallStartTime(new Date())
      setCallDuration(0)
      setFormData(prev => ({ 
        ...prev, 
        call_status: 'completed',
        call_duration: 0 
      }))
    } catch (error) {
      console.error('Error starting call:', error)
      alert('Failed to start call. Please try again.')
    }
  }

  const endCall = async () => {
    try {
      const token = localStorage.getItem('token')
      if (token) {
        // End call status
        await fetch('/api/user-status', {
          method: 'PUT',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`
          },
          body: JSON.stringify({
            action: 'end_call'
          })
        })
      }
    } catch (error) {
      console.error('Error ending call status:', error)
    }

    setIsCallActive(false)
    setCallStartTime(null)
  }

  const handleQuickCallback = (hours: number) => {
    const now = new Date()
    const callbackTime = new Date(now.getTime() + (hours * 60 * 60 * 1000))
    setFormData(prev => ({
      ...prev,
      callback_requested: true,
      callback_time: callbackTime.toISOString()
    }))
  }

  // Handle modal close - end call status if call was started
  const handleClose = async () => {
    if (isCallActive) {
      try {
        const token = localStorage.getItem('token')
        if (token) {
          await fetch('/api/user-status', {
            method: 'PUT',
            headers: {
              'Content-Type': 'application/json',
              'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
              action: 'end_call'
            })
          })
        }
      } catch (error) {
        console.error('Error ending call status on close:', error)
      }
    }
    onClose()
  }

  const validateForm = () => {
    const newErrors: Record<string, string> = {}

    if (!formData.notes.trim()) {
      newErrors.notes = 'Call notes are mandatory for all calls'
    }

    if (formData.callback_requested && !formData.callback_time) {
      newErrors.callback_time = 'Callback time must be specified when callback is requested'
    }

    setErrors(newErrors)
    return Object.keys(newErrors).length === 0
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    
    if (!validateForm()) return

    setIsLoading(true)
    try {
      await onSave(formData)
      
      // End call status when call is saved
      const token = localStorage.getItem('token')
      await fetch('/api/user-status', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({
          action: 'end_call'
        })
      })
      
      handleClose()
    } catch (error) {
      console.error('Error saving call log:', error)
    } finally {
      setIsLoading(false)
    }
  }

  const callStatusOptions = [
    { value: 'completed', label: 'Completed', icon: CheckCircleIcon, color: 'text-success-600' },
    { value: 'missed', label: 'Missed', icon: XCircleIcon, color: 'text-warning-600' },
    { value: 'declined', label: 'Declined', icon: XMarkIcon, color: 'text-danger-600' },
    { value: 'busy', label: 'Busy', icon: ExclamationTriangleIcon, color: 'text-warning-600' },
    { value: 'no_answer', label: 'No Answer', icon: ClockIcon, color: 'text-gray-600' }
  ]

  if (!isOpen) return null

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
        {/* Header */}
        <div className="p-6 border-b border-gray-200">
          <div className="flex justify-between items-start">
            <div>
              <h2 className="text-xl font-bold text-gray-900">
                {existingCallLog ? 'Edit Call Log' : 'Log Call'}
              </h2>
              <div className="mt-2 text-sm text-gray-600">
                <p className="font-medium">{client.principal_key_holder}</p>
                <p>{client.telephone_cell}</p>
                <p>Box: {client.box_number} | Contract: {client.contract_no}</p>
              </div>
            </div>
            <button
              onClick={handleClose}
              className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
            >
              <XMarkIcon className="w-5 h-5" />
            </button>
          </div>
        </div>

        <form onSubmit={handleSubmit} className="p-6 space-y-6">
          {/* Call Timer and Controls */}
          {!existingCallLog && (
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <div className="text-2xl font-mono font-bold text-gray-900">
                    {formatDuration(callDuration)}
                  </div>
                  {isCallActive && (
                    <div className="flex items-center text-green-600">
                      <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse mr-2"></div>
                      Call in progress
                    </div>
                  )}
                </div>
                <div className="flex space-x-2">
                  {!isCallActive ? (
                    <button
                      type="button"
                      onClick={startCall}
                      className="btn btn-success flex items-center"
                    >
                      <PlayIcon className="w-4 h-4 mr-2" />
                      Start Call
                    </button>
                  ) : (
                    <button
                      type="button"
                      onClick={endCall}
                      className="btn btn-danger flex items-center"
                    >
                      <StopIcon className="w-4 h-4 mr-2" />
                      End Call
                    </button>
                  )}
                </div>
              </div>
            </div>
          )}

          {/* Call Type */}
          <div>
            <label className="label">Call Type</label>
            <div className="flex space-x-4">
              <label className="flex items-center">
                <input
                  type="radio"
                  name="call_type"
                  value="outbound"
                  checked={formData.call_type === 'outbound'}
                  onChange={(e) => setFormData(prev => ({ ...prev, call_type: e.target.value as any }))}
                  className="mr-2"
                />
                Outbound
              </label>
              <label className="flex items-center">
                <input
                  type="radio"
                  name="call_type"
                  value="inbound"
                  checked={formData.call_type === 'inbound'}
                  onChange={(e) => setFormData(prev => ({ ...prev, call_type: e.target.value as any }))}
                  className="mr-2"
                />
                Inbound
              </label>
            </div>
          </div>

          {/* Call Status */}
          <div>
            <label className="label">Call Status</label>
            <div className="grid grid-cols-2 gap-3">
              {callStatusOptions.map((option) => (
                <label
                  key={option.value}
                  className={`flex items-center p-3 border rounded-lg cursor-pointer transition-colors ${
                    formData.call_status === option.value
                      ? 'border-primary-500 bg-primary-50'
                      : 'border-gray-300 hover:border-gray-400'
                  }`}
                >
                  <input
                    type="radio"
                    name="call_status"
                    value={option.value}
                    checked={formData.call_status === option.value}
                    onChange={(e) => setFormData(prev => ({ ...prev, call_status: e.target.value as any }))}
                    className="sr-only"
                  />
                  <option.icon className={`w-5 h-5 mr-3 ${option.color}`} />
                  <span className="font-medium">{option.label}</span>
                </label>
              ))}
            </div>
          </div>

          {/* Call Duration (if not using timer) */}
          {existingCallLog && (
            <div>
              <label className="label">Call Duration (seconds)</label>
              <input
                type="number"
                min="0"
                value={formData.call_duration || ''}
                onChange={(e) => setFormData(prev => ({ 
                  ...prev, 
                  call_duration: parseInt(e.target.value) || 0 
                }))}
                className="input"
                placeholder="Enter call duration in seconds"
              />
            </div>
          )}

          {/* Notes (Mandatory) */}
          <div>
            <label className="label">
              Call Notes <span className="text-danger-500">*</span>
            </label>
            <textarea
              value={formData.notes}
              onChange={(e) => setFormData(prev => ({ ...prev, notes: e.target.value }))}
              className={`input h-24 resize-none ${errors.notes ? 'border-danger-500' : ''}`}
              placeholder="Enter detailed notes about this call (mandatory)..."
              required
            />
            {errors.notes && (
              <p className="text-sm text-danger-600 mt-1">{errors.notes}</p>
            )}
          </div>

          {/* Callback Section */}
          <div className="bg-blue-50 rounded-lg p-4">
            <div className="flex items-center mb-3">
              <input
                type="checkbox"
                id="callback_requested"
                checked={formData.callback_requested}
                onChange={(e) => setFormData(prev => ({ 
                  ...prev, 
                  callback_requested: e.target.checked,
                  callback_time: e.target.checked ? prev.callback_time : ''
                }))}
                className="mr-3"
              />
              <label htmlFor="callback_requested" className="font-medium text-gray-900">
                Schedule Callback
              </label>
            </div>

            {formData.callback_requested && (
              <div className="space-y-3">
                {/* Quick Callback Buttons */}
                <div className="flex flex-wrap gap-2">
                  <button
                    type="button"
                    onClick={() => handleQuickCallback(1)}
                    className="btn btn-secondary btn-sm"
                  >
                    In 1 Hour
                  </button>
                  <button
                    type="button"
                    onClick={() => handleQuickCallback(2)}
                    className="btn btn-secondary btn-sm"
                  >
                    In 2 Hours
                  </button>
                  <button
                    type="button"
                    onClick={() => handleQuickCallback(24)}
                    className="btn btn-secondary btn-sm"
                  >
                    Tomorrow
                  </button>
                </div>

                {/* Custom Callback Time */}
                <div>
                  <label className="label">Custom Callback Time</label>
                  <input
                    type="datetime-local"
                    value={formData.callback_time ? new Date(formData.callback_time).toISOString().slice(0, 16) : ''}
                    onChange={(e) => setFormData(prev => ({ 
                      ...prev, 
                      callback_time: e.target.value ? new Date(e.target.value).toISOString() : ''
                    }))}
                    className={`input ${errors.callback_time ? 'border-danger-500' : ''}`}
                    min={new Date().toISOString().slice(0, 16)}
                  />
                  {errors.callback_time && (
                    <p className="text-sm text-danger-600 mt-1">{errors.callback_time}</p>
                  )}
                </div>
              </div>
            )}
          </div>

          {/* Form Actions */}
          <div className="flex justify-end space-x-3 pt-4 border-t border-gray-200">
            <button
              type="button"
              onClick={handleClose}
              className="btn btn-secondary"
              disabled={isLoading}
            >
              Cancel
            </button>
            <button
              type="submit"
              className="btn btn-primary"
              disabled={isLoading || !formData.notes.trim()}
            >
              {isLoading ? 'Saving...' : existingCallLog ? 'Update Call Log' : 'Save Call Log'}
            </button>
          </div>
        </form>
      </div>
    </div>
  )
}